<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Comment;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\Post\PostCommentTable;
use Erlage\Photogram\Data\Tables\Post\PostCommentLikeTable;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModel;

final class PostCommentContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(
                PostCommentTable::PARENT_POST_ID,
                RequestTable::PAYLOAD,
                PostCommentTable::TABLE_NAME
            );

            $offset = self::$request -> findKeyOffset(PostCommentTable::ID, PostCommentTable::TABLE_NAME);

            // optional
            $postCommentIdFromReq = self::$request -> findKey(
                PostCommentTable::REPLY_TO_POST_COMMENT_ID,
                RequestTable::PAYLOAD,
                PostCommentTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            $loadRepliesOnAPostComment = self::isAvailable($postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | check if post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post comment model exists(if loading replies)
            |--------------------------------------------------------------------------
            */

            if ($loadRepliesOnAPostComment)
            {
                $targetPostCommentModel = PostCommentModel::findFromId_throwException($postCommentIdFromReq);

                if ($targetPostModel -> getId() != $targetPostCommentModel -> getParentPostId())
                {
                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }
            }

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $postCommentTableQuery = (new Query()) -> from(PostCommentTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select where post id is matched
            |--------------------------------------------------------------------------
            */

            $postCommentTableQuery -> where(PostCommentTable::PARENT_POST_ID, $targetPostModel -> getId());

            /*
            |--------------------------------------------------------------------------
            | if loading replies
            |--------------------------------------------------------------------------
            */

            if ($loadRepliesOnAPostComment)
            {
                $postCommentTableQuery -> where(PostCommentTable::REPLY_TO_POST_COMMENT_ID, $targetPostCommentModel -> getId());
            }
            else
            {
                $postCommentTableQuery -> where(PostCommentTable::REPLY_TO_POST_COMMENT_ID, '0');
            }

            /*
            |--------------------------------------------------------------------------
            | post likes selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $postCommentTableQuery -> greaterThan(PostCommentTable::ID, $offset);
            }
            else
            {
                $postCommentTableQuery -> lessThan(PostCommentTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $postCommentTableQuery
                -> orderByDesc(PostCommentTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_POST_COMMENT));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $postCommentBeans = $postCommentTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($postCommentBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare post comment maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(PostCommentTable::getTableName(), $postCommentBeans, function (PostCommentModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's post comments are selected)
                |--------------------------------------------------------------------------
                */

                self::addDependency(UserTable::getTableName(), $model -> getOwnerUserId());
            });

            /*
            |--------------------------------------------------------------------------
            | additional data | post comment like maps
            | -------------------------------------------------------------------------
            | help build like button next to post comment
            |--------------------------------------------------------------------------
            */

            $postCommentsContainer = self::$dataDock -> getContainer(PostCommentTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                PostCommentLikeTable::getTableName(),
                array(
                    PostCommentLikeTable::LIKED_POST_COMMENT_ID => $postCommentsContainer -> getIds(),
                    PostCommentLikeTable::LIKED_BY_USER_ID      => self::$authedUserModel -> getId(),
                )
            );
        });
    }
}
